import java.io.*;
import java.net.*;



/*

	#########
	tcp_client class
	#########
	
	
	
	FUNCTION										USAGE
	###CONSTRUCTORS###
	tcp_client(String hostname, int port)						constructor; creates TCP socket session based on hostname and client
	
	###NET FUNCTIONS###
	connect()										establishes the connection
	close()											closes the connection
	command(String command)								sends a command to the socket and returns the answer
	
	###DEBUG FUNCTIONS###
	isClosed()										returns whether the socket is closed or not
	debugMsg(String msg)								writes a debug message

*/



public class tcp_client
/* class for implementing a simple tcp/ip connection */
{

	//host configuration
	String hostname, answer;
	int port;
	
	//socket variables
	Socket clientSocket;
	DataOutputStream outToServer;
	BufferedReader inFromServer;
	
	
	
	/****************
	CONSTRUCTORS
	****************/
	
	tcp_client(String hostname, int port) throws Exception
	/* constructor; creates TCP socket session based on hostname and client */
	{
		//set hostname and port class-wide
		this.hostname = hostname;
		this.port = port;
		
		//Try to connect
		this.connect();
	}
	
	
	
	/*****************
	NET FUNCTIONS
	*****************/
	
	private void connect() throws IOException
	/* establishes the connection */
	{
		//Try to create tcp/ip connection or die in a fire
		debugMsg("Create connection to " + this.hostname + ":" + this.port + "...");
		try{
			this.clientSocket = new Socket(this.hostname, this.port);
			
			this.clientSocket.setSoTimeout(10000);
			//this.clientSocket.setKeepAlive(false);
			this.clientSocket.setSendBufferSize(8192);
			this.clientSocket.setReceiveBufferSize(8192);
			
			this.outToServer = new DataOutputStream(clientSocket.getOutputStream());
			this.inFromServer = new BufferedReader(new InputStreamReader(clientSocket.getInputStream()));
			debugMsg("TCP connection to " + hostname + " at port " + port + " established.");
		}
		catch (SocketException e) { System.err.println ("Socket error : " + e); }
		catch (UnknownHostException e) { System.err.println ("Invalid host!"); }
		catch (IOException e) { System.err.println ("I/O error : " + e); }
	}
	
	public void close() throws IOException { clientSocket.close(); debugMsg("Closed socket"); }	//closes the connection
	
	public String command(String command) throws IOException, InterruptedException
	/* sends a command to the socket and returns the answer */
	{
		debugMsg("About to send command: '" + command + "'");
		debugMsg("SoLinger: " + clientSocket.getSoLinger());
		debugMsg("SoTimeout: " + clientSocket.getSoTimeout());
		debugMsg("sendBuffer: " + clientSocket.getSendBufferSize());
		debugMsg("receiveBuffer: " + clientSocket.getReceiveBufferSize());
		debugMsg("KeepAlive: " + clientSocket.getKeepAlive());
		debugMsg("isConnected: " + clientSocket.isConnected());
		debugMsg("isInputShutdown: " + clientSocket.isInputShutdown());
		debugMsg("isOutputShutdown: " + clientSocket.isOutputShutdown());
		
		//reconnecting if socket closed
		if(clientSocket.isClosed()) { debugMsg("Socket closed, reconnecting..."); this.connect(); }
		
		//send command and retrieve answer
		outToServer.writeBytes(command + '\n');
		debugMsg("Sent command: " + command);
		
		//wait a little moment before reading the answer
		Thread.sleep(500);
		
		//get the answer or die in a fire
		try { answer = inFromServer.readLine(); }
		catch(Exception e) { debugMsg("Error: " + e); }
		debugMsg("Got answer: " + answer);
		
		//return answer
		return answer;
	}
	
	
	
	/********************
	DEBUG FUNCTIONS
	********************/
	
	public boolean isClosed() { return this.clientSocket.isClosed(); }				//returns whether the socket is closed or not
	private void debugMsg(String msg) { System.out.println("TCP_CLIENT: " + msg); }	//writes a debug message

}