import java.io.*;
import java.net.*;



/*

	##########
	netio_client class
	##########
	
	
	
	FUNCTION										USAGE
	###CONSTRUCTORS###
	netio_client(String hostname, int port, String username, String password)	Constructor; new client based on hostname, port, username and password
	
	###BASIC FUNCTIONS###
	command(String command)								sends command
	login()											performs a login with normal password
	cryptlogin()										performs a "secure" login with passphrase
	logout()										performs a logout
	countChars(String s, char c)								prints the amount of equal char in a string
	
	###GET FUNCTIONS###
	getHostname()									gets the hostname
	getVersion()										...the firmware version
	getTime()										...the local time in format YYYY/MM/DD,HH:MM:SS
	getTimezone()										....the timezone offset
	getSNTPState()									...SNTP state (enabled/disabled)
	getSNTP()										...all SNTP information (IP address, enabled/disabled, synchronized)
	getPowerDelay()									...power delay before operations
	getPorts()										...an array for the ports with status (enabled/disabled)
	getPort(int port)									...state of a port
	getPortWatchdog(int port)								...all information of a defined port watchdog (status/ip address/timeout/delay/ping refresh/max retry/max retry poweroff/send mail)
	getSystemDiscover()									...state of system discover function
	getNetwork()										...all network information (configuration method/ip address/netmask/gateway/DNS/NTP)
	getDNS()										...configured DNS server
	getMailserver()									...configured mail server
	
	###SET FUNCTIONS###
	setHostname(String name)								sets the hostname
	setTime(String time)									...the local time in format YYYY/MM/DD,HH:MM:SS
	setTimezone(String offset)								...the timezone offset
	setSNTP(String address)								...SNTP server
	enableSNTP()										enables SNTP for time synchronization
	disableSNTP()									disables SNTP
	setPowerDelay(int delay)								sets the delay delay before operations
	configurePort(int port, String name, boolean useTimer,				configures a port
	int delay, boolean poweredOn)
	setPortActive(int port)								enables a port
	setPortInactive(int port)								disables a port
	configureTimer()									TODO
	configureWatchdog(int port, String address, int timeout,				configures a port's watchdog
	int pon_delay, int ping_interval, int max_retry,
	boolean max_retry_poff, boolean send_mail)
	enableWatchdog(int port)								enables the watchdog of a specified port
	disableWatchdog(int port)								disables the watchdog of a specified port
	enableDiscover()									enables the discovery functionality
	disableDiscover()									disables the discovery functionality
	setNetwork(String address, String netmask, String gateway)			configures the network based on given address, netmask and gateway
	enableDHCP()										enables DHCP for network auto-configuration
	setDNS(String address)								sets the DNS server
	setMailserver(String address)							configures the mailserver
	
	###KILL FUNCTIONS###
	deviceReboot()									reboots the NETIO device immediately
	hardReset()										resets the whole NETIO device configuration
	
	###DEBUG FUNCTIONS###
	setTelnetFix()									enables/disables the telnet fix

*/



public class netio_client
/* class for controlling a NETIO client through a KShell session */
{
	
	//variables
	tcp_client tcpNetio;
	String hostname, username, password;
	int port;
	boolean cryptlogin = false;
	
	//runtime variables
	boolean loggedIn = false;
	boolean useTelnetFix = true;
	
	
	
	netio_client(String hostname, int port, String username, String password, boolean cryptlogin) throws Exception
	/* creates a new client based on given hostname, port, username and password */
	{
		//Set username and password for class
		this.hostname = hostname;
		this.port = port;
		this.username = username;
		this.password = password;
		this.cryptlogin = cryptlogin;
		debugMsg("New NETIO client with hostname='" + hostname + "', port='" + port + "', username='" + username + "', password='" + password + "', cryptlogin='" + cryptlogin + "'");
	}
	
	
	
	/*******************
	BASIC FUNCTIONS
	*******************/
	
	public String command(String command) throws Exception
	/* send command */
	{
		String result = "";
		debugMsg("Command: '" + command + "'");
		//if(this.loggedIn == false && this.useTelnetFix == true) { login(); }
		if(this.loggedIn == false) { login(); }
		//login();
		try{
			result = tcpNetio.command(command);
		}
		catch(Exception e)
		{
			this.loggedIn = false;
			this.tcpNetio.close();
		}
		if(this.useTelnetFix == true) { logout(); }
		return result;
	}
	
	public boolean login() throws Exception
	/* Login using username and password */
	{
		debugMsg("Re-Login");
		//Try to create tcp/ip connection or die in a fire
		try
		{
			this.tcpNetio = new tcp_client(this.hostname, this.port);
			
			tcpNetio.command("");
			//tcpNetio.command("");
			
			//Try to login
			String temp;
			if(this.cryptlogin == true) { temp = tcpNetio.command("clogin " + this.username + " " + this.password); }
			else { temp = tcpNetio.command("login " + this.username + " " + this.password); }
			
			//Return result
			if(temp.substring(0, 6).equals("250 OK"))
			{
				this.loggedIn = true;
				System.out.println("loggedIn: " + this.loggedIn);
				return true;
			}
			else
			{
				this.loggedIn = false;
				System.out.println("loggedIn: " + this.loggedIn);
				return false;
			}
		
		}
		catch (SocketException e ) { System.err.println ("Socket error : " + e); this.loggedIn=false; return false; }
		catch (UnknownHostException e ) { System.err.println ("Invalid host!"); this.loggedIn=false; return false; }
		catch (IOException e ) { System.err.println ("I/O error : " + e); this.loggedIn=false; return false; }
	}
	
	public void logout() throws Exception
	{
		tcpNetio.command("quit");
		this.loggedIn = false;
		tcpNetio.close();
	}
	private static int countChars(String s, char c) { return s.replaceAll("[^" + c + "]","").length(); }		//count equal chars in a string
	
	
	
	/*****************
	GET FUNCTIONS
	*****************/
	
	public String getHostname() throws Exception { return command("alias").substring(4); }				//get hostname
	public String getVersion() throws Exception { return command("version").substring(6); }				//get version
	public String getTime() throws Exception { return command("system time").substring(4); }			//get time
	public String getTimezone() throws Exception { return command("system timezone").substring(4); }		//get timezone
	public boolean getSNTPState() throws Exception { if(command("system sntp").contains("enable")) { return true; } else { return false; } }	//get state of SNTP
	
	public String[] getSNTP() throws Exception
	/* get all sntp information */
	{
		//define array and get command result
		String[] sntp = new String[3];
		String command = command("system sntp");
		
		//set enabled flag if sntp enabled and cut string
		if(command.contains("enable")) { sntp[1] = "true"; } else { sntp[1] = "false"; }
		command = command.replace("enable ", "");
		command = command.replace("disable ", "");
		
		//set synchronized flag if sntp is synchronized and cut string
		if(command.contains("synchronized")) { sntp[2] = "true"; } else { sntp[2] = "false"; }
		command = command.replace(" synchronized", "");
		command = command.substring(4);
		
		//set SNTP ip address and return array
		sntp[0] = command;
		return sntp;
	}
	
	public String getPowerDelay() throws Exception { return command("system swdelay").substring(4); }		//get power delay
	
	//public boolean[] getPorts() throws Exception
	public String[][] getPorts() throws Exception
	/* get all port states and their configuration */
	{
		//boolean[] ports = new boolean[4];
		String[][] ports = new String[4][4];
		String command = command("port list").substring(4);
		String subcommand;
		for(int i=0;i<4;i++)
		{
			/*if(command.substring(i, i+1).equals("1"))
			{ ports[i] = true; } else { ports[i] = false; }*/
			
			if(command.substring(i, i+1).equals("1")) { ports[i][0] = "true"; }
			else { ports[i][0] = "false"; }
			
			//get port name, timer/manual, delay
			subcommand = command("port setup " + (i+1)).substring(5);		//4 = "
			ports[i][1] = subcommand.substring(0, subcommand.indexOf('"'));
			
			//if(subcommand.contains("manual")) { ports[i][2] = "manual"; } else { ports[i][2] = "timer"; }
			if(command.contains("manual")) { ports[i][2] = "manual"; } else { ports[i][2] = "timer"; }
			
			//ports[i][3] = subcommand.substring(subcommand.length()-1, subcommand.length());
			ports[i][3] = subcommand.substring(subcommand.length()-3, subcommand.length()-2);
			
		}
		return ports;
	}
	
	public boolean getPort(int port) throws Exception
	/* get port state of specified port */
	{
		if(command("port " + port).substring(4,5).equals("1")) { return true; }
		else { return false; }
	}
	
	public String[] getPortWatchdog(int port) throws Exception
	/* get all port watchdog information of a specified port */
	{
		//define array and get command result
		String[] thisPort = new String[8];
		String command = command("port wd " + port).substring(4);
		
		//active/inactive
		thisPort[0] = command.substring( 0, command.indexOf(" ")  );
		command = command.substring(command.indexOf(" ")+1);
		
		//cut IP address
		thisPort[1] = command.substring( 0, command.indexOf(" ")  );
		command = command.substring(command.indexOf(" ")+1);
		
		//cut timeout
		thisPort[2] = command.substring( 0, command.indexOf(" ")  );
		command = command.substring(command.indexOf(" ")+1);
		
		//cut poweron delay
		thisPort[3] = command.substring( 0, command.indexOf(" ")  );
		command = command.substring(command.indexOf(" ")+1);
		
		//cut ping refresh
		thisPort[4] = command.substring( 0, command.indexOf(" ")  );
		command = command.substring(command.indexOf(" ")+1);
		
		//cut max retry
		thisPort[5] = command.substring( 0, command.indexOf(" ")  );
		command = command.substring(command.indexOf(" ")+1);
		
		//cut max retry poff
		thisPort[6] = command.substring( 0, command.indexOf(" ")  );
		command = command.substring(command.indexOf(" ")+1);
		
		//cut send mail
		thisPort[7] = command;
		
		//return array
		return thisPort;
	}
	
	public boolean getSystemDiscover() throws Exception
	/* get status of system discovery service */
	{
		if(command("system discover").substring(4).equals("enable")) { return true; }
		else { return false; }
	}
	
	public String[] getNetwork() throws Exception
	/* get complete network configuration */
	{
		//define array and get command result
		String network[] = new String[6];
		String command = command("system eth").substring(4);
		
		//cut first param (dhcp/static)
		network[0] = command.substring( 0, command.indexOf(" ")  );
		command = command.substring(command.indexOf(" ")+1);
		
		//cut IP address
		network[1] = command.substring( 0, command.indexOf(" ")  );
		command = command.substring(command.indexOf(" ")+1);
		
		//cut netmask
		network[2] = command.substring( 0, command.indexOf(" ")  );
		command = command.substring(command.indexOf(" ")+1);
		
		//cut gateway
		network[3] = command;
		
		//DNS
		//network[4] = command("system dns").substring(4);
		network[4] = getDNS();
		
		//NTP
		network[5] = getSNTP()[0];
		
		//Return array
		return network;
	}

	public String getDNS() throws Exception { return command("system dns").substring(4); }				//get dns
	public String getMailserver() throws Exception { return command("email server").substring(4); }		//get mail server
	
	
	
	/*****************
	SET FUNCTIONS
	*****************/
	
	public boolean setHostname(String name) throws Exception
	/* sets the hostname */
	{
		//set hostname if longer than 2 chars
		if(name.length() > 2)
		{
			//execute command and return result
			if(command("alias " + name).substring(0, 6).equals("250 OK")) { return true; }
			else { return false; }
		}
		else { return false; }
	}
	
	public boolean setTime(String time) throws Exception
	/* sets the time in format YYYY/MM/DD,HH:MM:SS */
	{
		if((time.length() == 18 || time.length() == 19) && countChars(time, '/') == 2 && countChars(time, ':') == 2 && countChars(time, ',') == 1)
		{
			if(command("system time " + time).substring(0, 6).equals("250 OK")) { return true; }
			else { return false; }
		}
		else { return false; }
	}
	
	public boolean setTimezone(String offset) throws Exception
	/* sets the timezone using +/- offset */
	{
		if((offset.length() <= 3) && (offset.substring(0, 1).equals("+") || offset.substring(0, 1).equals("-")))
		{
			if(command("system timezone " + offset).substring(0, 6).equals("250 OK")) { return true; }
			else { return false; }
		}
		else { return false; }
	}
	
	public boolean setSNTP(String address) throws Exception
	/* configures SNTP */
	{
		if((address.length() >= 8 && address.length() <= 15) && countChars(address, '.') == 3)
		{
			if(command("system sntp enable " + address).substring(0, 6).equals("250 OK")) { return true; }
			else { return false; }
		}
		else { return false; }
	}
	
	public boolean enableSNTP() throws Exception
	/* enables SNTP */
	{
		if(command("system sntp enable").substring(0, 6).equals("250 OK")) { return true; }
		else { return false; }
	}
	
	public boolean disableSNTP() throws Exception
	/* disables SNTP */
	{
		if(command("system sntp disable").substring(0, 6).equals("250 OK")) { return true; }
		else { return false; }
	}
	
	public boolean setPowerDelay(int delay) throws Exception
	/* set power delay */
	{
		if(delay > 1 && delay < 6000)
		{
			if(command("system swdelay " + delay).substring(0, 6).equals("250 OK")) { return true; } else { return false; }
		}
		else { return false; }
	}
	
	
	
	public boolean configurePort(int port, String name, boolean useTimer, int delay, boolean poweredOn) throws Exception
	/* configures a port */
	{
		//variables
		String str_mod, str_pon;
		
		if(port >= 1 && port <= 4)
		{
			if(delay > 2) { delay = 2; }
			if(useTimer == true) { str_mod = "timer"; } else { str_mod = "manual"; }
			if(poweredOn == true) { str_pon = "enable"; } else { str_pon = "disable"; }
			//execute command and return result
			if(command("port setup " + port + " " + name + " " + str_mod + " " + delay + " " + str_pon).substring(0, 6).equals("250 OK"))
			{ return true; } else { return false; }
		}
		else { return false; }
	}
	
	public void setPortActive(int port) throws Exception { command("port " + port + " 1").substring(4); }	//enable port
	public void setPortInactive(int port) throws Exception { command("port " + port + " 0").substring(4); }	//disable port
	
	public boolean configureTimer()
	/* configures the timer */
	{
		/*
			port timer <output> <time_format> [ <mode: once | daily | weekly> <on-time> <off-time>] <week_schedule>
			  Timer control: 
			  < output_number> - number of output to change 
			<time_format> - Zeitformat 
			t  - HH:MM:SS                                                 
			dt - YYYY/MM/DD,HH:MM:SS 
			ux - xxxxxxxx ( unsigned long mit 0x<hex>, 0<octal> prefix oder decadical)   
			<week_schedule> - 7-Bit-Zahl (0 or 1). Das erste Digit steht für Montag und das letzte für Sontag. 
			 
			Beispiel: Der Befehl port timer 3 t weekly 08:00:00 17:30:00 1111100 setzt Time control on output 3 
			auf Enable. Jeder Tag von Montag bis Freitag 08:00 Ausgang 3 wird auf ON gesetzt und auf 
			OFF um 17:30
		*/
		
		//Ehh?
		return false;
	}
	
	public boolean configureWatchdog(int port, String address, int timeout, int pon_delay, int ping_interval, int max_retry, boolean max_retry_poff, boolean send_mail) throws Exception
	/* configures the port watchdog */
	{
		//variables
		String str_max_retry_poff, str_send_mail;
		if(max_retry_poff == true) { str_max_retry_poff = "enable"; } else { str_max_retry_poff = "disable"; }
		if(send_mail == true) { str_send_mail = "enable"; } else { str_send_mail = "disable"; }
		
		//configure the port watchdog if address valid
		if(
			((address.length() >= 8 && address.length() <= 15) && countChars(address, '.') == 3) &&
			(port >= 1 && port <= 4)
		)
		{
			//execute command and return result
			if(command("port wd " + port + " enable " + address + " " + timeout + " " + pon_delay + " " + ping_interval + " " + max_retry + " " + str_max_retry_poff + " " + str_send_mail).substring(0, 6).equals("250 OK"))
			{ return true; } else { return false; }
		}
		else { return false; }
	}
	
	public boolean enableWatchdog(int port) throws Exception
	/* enables the watchdog of a specified port */
	{
		if(port >= 1 && port <= 4)
		{
			if(command("port wd " + port + " enable").substring(0, 6).equals("250 OK")) { return true; } else { return false; }
		}
		else { return false; }
	}
	
	public boolean disableWatchdog(int port) throws Exception
	/* disables the watchdog of a specified port */
	{
		if(port >= 1 && port <= 4)
		{
			if(command("port wd " + port + " disable").substring(0, 6).equals("250 OK")) { return true; } else { return false; }
		}
		else { return false; }
	}
	
	
	
	public boolean enableDiscover() throws Exception
	/* enables the discover functionality */
	{
		if(command("system discover enable").substring(0, 6).equals("250 OK")) { return true; } else { return false; }
	}
	
	public boolean disableDiscover() throws Exception
	/* disables the discover functionality */
	{
		if(command("system discover disable").substring(0, 6).equals("250 OK")) { return true; } else { return false; }
	}
	
	
	
	public boolean setNetwork(String address, String netmask, String gateway) throws Exception
	/* configures the network */
	{
		//execute command if addresses valid
		if(
			((address.length() >= 8 && address.length() <= 15) && countChars(address, '.') == 3) &&
			((netmask.length() >= 8 && netmask.length() <= 15) && countChars(netmask, '.') == 3) &&
			((gateway.length() >= 8 && gateway.length() <= 15) && countChars(gateway, '.') == 3)
		)
		{
			//execute command and return result
			if(command("system eth manual " + address + " " + netmask + " " + gateway).substring(0, 6).equals("250 OK")) { return true; }
			else { return false; }
		}
		else { return false; }	//invalid addresses
	}
	
	public boolean enableDHCP() throws Exception
	/* enable DHCP */
	{
		if(command("system eth dhcp").substring(0, 6).equals("250 OK")) { return true; } else { return false; }
	}
	
	public boolean setDNS(String address) throws Exception
	/* sets the dns server */
	{
		if((address.length() >= 8 && address.length() <= 15) && countChars(address, '.') == 3)
		{
			if(command("system dns " + address).substring(0, 6).equals("250 OK")) { return true; } else { return false; }
		}
		else { return false; }
	}
	
	public boolean setMailserver(String address) throws Exception
	/* sets mail server address or hostname */
	{
		if(command("email server " + address).substring(0, 6).equals("250 OK")) { return true; } else { return false; }
	}
	
	
	
	/*****************
	KILL FUNCTIONS
	*****************/	
	
	public void deviceReboot() throws Exception { command("reboot"); }
	public void hardReset() throws Exception { command("system reset to default"); }
	
	
	
	/********************
	DEBUG FUNCTIONS
	********************/
	
	public void setTelnetFix(boolean value)
	//enables/disables the telnet fix
	{ this.useTelnetFix = value; }
	
	public boolean getTelnetFix() { return this.useTelnetFix; }		//returns whether the telnet fix is used or not
	public boolean isLoggedIn() { return this.loggedIn; }			//returns whether the connection is established or not
	
	private void debugMsg(String msg) { System.out.println("NETIO_CLIENT: " + msg); }	//writes a debug message

}